/****************************************************************************
 * 
 * Class:  Coordinator implementation
 * Author: Mark Roseman
 * 
 * Revision History:
 * 
 * Date     Modifier  Description
 * -------- --------- -------------------------------------------------------
 * 02/19/92 MR        initial version
 * 06/08/92 MR        extended to create conferences as separate processes
 * 09/17/92 MR        added callback for reader connection close
 * 09/20/92 MR        changed createConference to do more flexible parameter
 *                      passing (i.e. no longer order dependent)
 * 09/20/92 MR        added "userID_" field to ConferenceInfo, fixed so that
 *                      when a conference crashes we notify the registrar
 *                      client.  Also added UIDTbl for user id management.
 * 
 ****************************************************************************/

/*
 *  This file is part of GroupKit.
 *
 *  (c) Copyright 1992 Department of Computer Science, University of
 *      Calgary, Calgary, Alberta, Canada.  All rights reserved.
 *    
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appears in all copies.  The University
 *  of Calgary makes no representations about the suitability of this
 *  software for any purpose.  It is provided "as is" without express or
 *  implied warranty.
 */

#include <InterViews/style.h>
#include <gk/groupkit.h>
#include <gk-reg/coordinator.h>
#include <stdio.h>
#include <string.h>
#include <gk/reader.h>
#include <gk/writer.h>
#include <OS/host.h>
#include <gk/connection.h>
#include <OS/list.h>
#include <OS/table.h>
#include <OS/string.h>
#include <gk/straction.h>
#include <gk/connaction.h>
#include <gk/rpcaction.h>
#include <gk-reg/regclient.h>
#include <gk/infoconn.h>
#include <gk/msgsender.h>
#include <osfcn.h>
#include <IV-look/kit.h>
#include <Dispatch/rpcstream.h>
#include <gk/groupsession.h>


/*****************************************************************************
 * 
 * we need to retain info about the conferences that are hooked up to us
 * 
 *****************************************************************************/

class ConferenceInfo {
public:
  ConferenceInfo(Connection*, char*, int);
  Connection* conn_;
  char host_[80];
  int port_;
  int userID_;
};

ConferenceInfo::ConferenceInfo(Connection* conn, char* host, int port) : conn_(conn), port_(port) { strcpy(host_, host); }




/****************************************************************************
 *
 * Various callbacks, tables, etc.
 *
 ****************************************************************************/

declareConnActionCallback(Coordinator);
implementConnActionCallback(Coordinator);

declareStrActionCallback(Coordinator);
implementStrActionCallback(Coordinator);

declareRpcActionCallback(Coordinator);
implementRpcActionCallback(Coordinator);

declareTable(ConfTbl,int,ConferenceInfo*);
implementTable(ConfTbl,int,ConferenceInfo*);

declarePtrList(MsgQ, StrMsgSender);
implementPtrList(MsgQ, StrMsgSender);

// table for pending message queues
declareTable(MsgQTbl, int, MsgQ*);
implementTable(MsgQTbl, int, MsgQ*);

// table mapping conference id to usernum
declareTable(UIDTbl, int, int);
implementTable(UIDTbl, int, int);

/****************************************************************************
 *
 * Constructor.
 *
 ****************************************************************************/

Coordinator::Coordinator(Style* style) : RpcPeer("/dev/null"), style_(style) {
  confs_ = new ConfTbl(CONFTBLSIZE);
  pending_ = new MsgQTbl(20);
  uids_ = new UIDTbl(10);
  startListening();
}


/****************************************************************************
 *
 * Destructor.
 *
 ****************************************************************************/

Coordinator::~Coordinator() {
  delete confs_;
}


/***************************************************************************
 *
 * accept connections from our conferences that we spawned.
 * note: we'll get back a pointer to it in info-callback -- should probably
 * keep it around in a list for garbage collection purposes though..
 * 
 *****************************************************************************/

void Coordinator::createReaderAndWriter(int fd) {
  InfoConnection* in = new InfoConnection( fd, -1, nil, INFOMSG, 
					  new ConnActionCallback(Coordinator)
					  (this, &Coordinator::info_callback));
  in->reader()->closeCallback( new RpcActionCallback(Coordinator)
			      (this, &Coordinator::closeCallback));
}


/****************************************************************************
 *
 * info callback.  we've received the conference id from the conference,
 * as well as its address.  insert an entry in our table, and then
 * send any messages that have been pending to the conference.
 *
 ****************************************************************************/

void Coordinator::info_callback(char *s, class Connection* c) {
  int id, port, userid;
  char idstr[80], portstr[80];
  char host[80];
  MsgQ* msgs;
  ConferenceInfo* conf;

  AttributeList* al = AttributeList::read(s);
  al->find_attribute("confnum", idstr); id = atoi(idstr);
  al->find_attribute("host", host );
  al->find_attribute("port", portstr); port = atoi(portstr);

  c->id_ = id;
  confs_->insert(id, new ConferenceInfo(c, host, port) );

  if ( pending_->find(msgs, id) ) {
    for ( ListItr(MsgQ) i(*msgs); i.more(); i.next() ) {
      StrMsgSender* msg = i.cur();
      if(confs_->find(conf, id))
	msg->sendOn( conf->conn_->writer() );
      else
	printf("COORD: couldn't find confinfo\n");
    }
    pending_->remove(id);
  } else 
    fprintf(stderr, "COORD: couldn't find msgq id =%d\n", id);

  if (uids_->find(userid, id) ) {
    if(confs_->find(conf, id))
      conf->userID_ = userid;
    else
      fprintf(stderr, "Coordinator: couldn't find ConfInfo for userID_\n");
  }
}


/****************************************************************************
 *
 * create a conference of the given type.  set up the command line arguments,
 * fork a new process, and then transfer its control to the conference.
 *
 ****************************************************************************/

boolean Coordinator::createConference(AttributeList* al) {
  extern char **environ;
  String execname;
  String desc;
  String bindir;
  char att[50], name[80], type[80], conf[80];

  al->find_attribute("name", name);
  al->find_attribute("type", type);
  al->find_attribute("confnum", conf);
  
  /*
   * look up name of executable in x defaults 
   */

  long types;
  if( style_->find_attribute( "conferenceTypes", types)) {
    for(int i = 1; i <= types; i++) {
      sprintf(att, "conf%d-desc", i);
      if (style_->find_attribute( att, desc )) {
	if( desc == type ) {
	  sprintf(att, "conf%d-prog", i);
	  if (!style_->find_attribute( att, execname ))
	    fprintf(stderr,"Coordinator: couldn't get conf-prog %d.\n", i);
	}
	
      } else
	fprintf(stderr,"Coordinator: couldn't get conf-desc %d.\n", i);
    }
  } else
    fprintf(stderr,"Coordinator: no conferenceTypes resource.\n");

  
  pending_->insert( atoi(conf), new MsgQ);

  /*
   * set up command line arguments and run process
   */

  style_->find_attribute( "GroupKitBinDir", bindir);
  char s[1000], attrrep[1000];
  al->write(attrrep);
  sprintf( s, "%s/%s -confname \'%s\' -confid %s -coordhost %s -coordport %d -attrs \'%s\' &",
	  bindir.string(), execname.string(), name, conf, 
	  GroupSession::host_name(), _lPort, attrrep);
  int err = system(s);
  if (err!=0)
    fprintf(stderr, "Coordinator: after system(), error was %d\n", err);
  return true;
}


/****************************************************************************
 *
 * Send a message to a particular conference.  If the conference has not
 * yet connected to us, queue the message up for later transmission.
 *
 ****************************************************************************/

void Coordinator::toConf(int confnum, StrMsgSender* msg) {
  ConferenceInfo* conf;
  if(confs_->find(conf, confnum))
    msg->sendOn( conf->conn_->writer() );
  else {
    MsgQ* msgs;
    if(pending_->find(msgs, confnum))
      msgs->append ( msg ); 
  }
}


/****************************************************************************
 *
 * delete the given conference number.
 *
 ****************************************************************************/

void Coordinator::deleteConference(int confnum) {
  toConf ( confnum, new StrMsgSender ( DELCONF, " " ) );
}


/****************************************************************************
 *
 * connect up to a user
 * note: the address we're getting here is the address of the remote users
 *       coordinator, not the conference object.  so what we have to do is
 *       ask that coordinator for the address, then when its returned (via
 *       an AddrResp message) we can tell our conference to connect up to it 
 *
 ****************************************************************************/

void Coordinator::joinTo(AttributeList* al) {
  char s[200], host[80], port[80];
  al->find_attribute("host", host);
  al->find_attribute("port", port);
  al->write(s);
  InfoConnection* conn_ = new InfoConnection( host, atoi(port), -1, nil);

  if(conn_->writer()->server()) {
    conn_->callbacks()->insert(ADDRRESP, new StrActionCallback(Coordinator)
			       (this, &Coordinator::addrResp));
    conn_->reader()->closeCallback( new RpcActionCallback(Coordinator)
				   (this, &Coordinator::closeCallback));
    conn_->writer()->sendMsg( ADDRREQ, s );
  } else {
    fprintf(stderr, "Coordinator: could not hook up to remote coordinator ");
    fprintf(stderr, "for address resolution,\n");
    fprintf(stderr, "             Host=%s port=%s.\n", host, port);
    fprintf(stderr, "             We'll continue running for now.\n");
  }
}


/****************************************************************************
 *
 * We've received a request from another coordinator for the address of one
 * of our conferences.  Find out the address, and then transmit back an
 * "address-response" message with the details.  Two things to fix:
 * 
 *     - should queue up messages when conference not hooked up yet
 *     - should get rid of connection after sending message
 *
 ****************************************************************************/

void Coordinator::addrReq(char *s, Connection* c)  { 
  char t[200], confnum[80], port[80];
  ConferenceInfo* conf;
  AttributeList* al = AttributeList::read(s);
  al->find_attribute( "confnum", confnum );
  if(confs_->find(conf, atoi(confnum))) {
    al->attribute( "host", conf->host_ );
    sprintf(port, "%d", conf->port_);
    al->attribute( "port", port);
    al->write(t);
    c->writer()->sendMsg ( ADDRRESP, t );
    //  delete c;
  } else
    printf("Coordinator: cannot find conference %d\n", confnum);
}

/****************************************************************************
 *
 * We've received the address of a conference that we want to connect up to.
 * Tell the local conference to connect to the newly found remote conference.
 *
 ****************************************************************************/

void Coordinator::addrResp(char *s) { 
  char confnum[80];
  AttributeList* al = AttributeList::read(s);
  al->find_attribute("confnum", confnum);
  toConf( atoi(confnum), new StrMsgSender ( CONNECTTO, s ) );
}


/****************************************************************************
 *
 * delete a user from a conference
 *
 ****************************************************************************/

void Coordinator::deleteUser(int confnum, int usernum) {
  char s[80];
  sprintf(s, "usernum=%d", usernum);
  toConf ( confnum, new StrMsgSender ( DELUSER, s ) );
}


/****************************************************************************
 *
 * We've received information for one of our conferences (e.g. ID number).
 * Pass it on.  If conference isn't hooked up, store it in the uids_ table,
 * where it will get picked up by the info_callback.
 *
 ****************************************************************************/

void Coordinator::setLocalInfo(AttributeList* al) {
  char s[200], confnum[80], usernum[80];
  al->find_attribute( "confnum", confnum );
  al->find_attribute( "usernum", usernum );
  ConferenceInfo* conf;
  al->write(s);
  if(confs_->find(conf, atoi(confnum)))
    conf->userID_ = atoi(usernum);
  else
    uids_->insert( atoi(confnum), atoi(usernum));
  toConf ( atoi(confnum), new StrMsgSender ( YOUAREID, s ) );
}


/****************************************************************************
 *
 * One of the connected conferences has died.
 *
 ****************************************************************************/

void Coordinator::closeCallback(class CallbackRpcReader* r, int /* fd */) {
  for ( TableIterator(ConfTbl) i(*confs_); i.more(); i.next() ) {
    if ( i.cur_value()->conn_->reader() == r ) {
//      fprintf(stderr, "Coordinator: connection to conference id=%d closed\n",
//      i.cur_value()->conn_->id_);
      rc_->userLeft( i.cur_key(), i.cur_value()->userID_);
      confs_->remove( i.cur_key() );
      break;
    }
  }
}



