/**************************************************************************
 * 
 * Class:  CursorOverlay implementation
 * Author: Mark Roseman
 * 
 * Revision History:
 * 
 * Date     Modifier  Description
 * -------- --------- -------------------------------------------------------
 * 05/28/92 MR        initial version
 * 10/22/92 MR        added stuff for colors, etc.
 *
 **************************************************************************/

/*
 *  This file is part of GroupKit.
 *
 *  (c) Copyright 1992 Department of Computer Science, University of
 *      Calgary, Calgary, Alberta, Canada.  All rights reserved.
 *    
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appears in all copies.  The University
 *  of Calgary makes no representations about the suitability of this
 *  software for any purpose.  It is provided "as is" without express or
 *  implied warranty.
 */

#include <stdio.h>
#include <gk/cursor.h>
#include <InterViews/style.h>
#include <gk/conference.h>
#include <gk/infoconn.h>
#include <OS/table.h>
#include <OS/string.h>
#include <InterViews/patch.h>
#include <InterViews/canvas.h>
#include <InterViews/window.h>
#include <IV-look/kit.h>
#include <InterViews/cursor.h>
#include <gk/straction.h>
#include <gk/reader.h>
#include <gk/groupkit.h>
#include <gk/msgsender.h>
#include <InterViews/page.h>
#include <InterViews/bitmap.h>
#include <InterViews/stencil.h>
#include <InterViews/color.h>
#include <InterViews/event.h>
#include <InterViews/input.h>
#include <InterViews/session.h>
#include "cursor.bit"
#include <InterViews/hit.h>
#include <gk/attrlist.h>
#include <stdlib.h>

declareTable(IndexTbl, int, GlyphIndex)
implementTable(IndexTbl, int, GlyphIndex)

declareStrActionCallback(CursorOverlay);
implementStrActionCallback(CursorOverlay);



/**************************************************************************
 * 
 * Constructor.  Stick the background glyph in a page.  Create a table
 * which contains ID->GlyphIndex mappings.  Add a callback for the 
 * cursor movement message, and then install ourselves as a conference
 * monitor so we receive new user and user leaving messages.
 *
 **************************************************************************/

CursorOverlay::CursorOverlay(ActiveHandler* background, Style* style, 
			     Conference* conf) : 
      Overlay(background, style, conf) 
{
  pg_ = new Page(background);
  patch_ = new Patch(pg_);
  body( patch_ );
  tbl_ = new IndexTbl(20);
  conf->connections()->callbacks()->insert( CURSORMOVE, 
					   new StrActionCallback(CursorOverlay)
					   (this, &CursorOverlay::remoteMove));
  conf->monitors()->append( this);

  localBitmapCursor = style->value_is_on("CursorOverlay-localBitmapCursor");
  localRealCursor = style->value_is_on("CursorOverlay-localRealCursor");
  window_cursor_ = nil;
}


/**************************************************************************
 * 
 * A new user has arrived, so create a new cursor symbol and insert it
 * into the Page glyph.  Update the table.
 *
 **************************************************************************/

void CursorOverlay::newUser(AttributeList* al) {
  char id[80];		   
  String cursor_color;
  if (al->find_attribute("usernum", id)) {
    Bitmap* bm = new Bitmap(cursor_bits, 32, 32);
    style()->find_attribute( "CursorOverlay-foreground", cursor_color );
    const Color* clr = Color::lookup( Session::instance()->default_display(), 
			       cursor_color );
    Stencil* st = new Stencil(bm, (clr==nil? new Color(1,1,1) : clr) );
    pg_->append(st);
    pg_->show (pg_->count() -1, false);
    patch_->reallocate();
    patch_->redraw();
    tbl_->insert( atoi(id), pg_->count()-1 );
  } else {
    char s[1000];
    al->write(s);
    fprintf(stderr, "Cursor: could not get id <%s>\n",s);
  }
}


/**************************************************************************
 * 
 * A user is leaving, so delete the glyph from the Page and update the
 * table.
 *
 **************************************************************************/

void CursorOverlay::userLeaving(int id) {
  tbl_->remove(id);
  /* have to muck with table after deleting glyph */
  for( TableIterator(IndexTbl) i(*tbl_); i.more(); i.next()) {
    if (i.cur_value() > id)
      (i.cur_value())--;
  }
}


/**************************************************************************
 * 
 * Move a particular cursor.  Don't display bitmap if "localBitmapCursor"
 * is false.
 *
 **************************************************************************/

void CursorOverlay::moveCursor(int id, int x, int y) {
  GlyphIndex idx;
  if(tbl_->find( idx, id)) 
    if( (x==-1) && (y==-1) ) 
      pg_->show( idx, false );
    else {
      if ( (localBitmapCursor && (id==conference()->localID())) || 
	   (id!=conference()->localID()))
	pg_->show( idx, true );
      pg_->move( idx, x-pg_->x()-cursor_x_hot, y-pg_->y()-cursor_y_hot );
    }
  else
    fprintf(stderr, "...could not find in table id=%d\n", id);
}


/**************************************************************************
 * 
 * Callback that a remote user has moved his cursor.  Note we're just
 * using sprintf/sscanf rather than attribute lists for transmitting.
 *
 **************************************************************************/

void CursorOverlay::remoteMove(char *s) {
  int id, x, y;
  sscanf(s, "%d:%d:%d", &id, &x, &y);
  moveCursor(id,x,y);
}


/**************************************************************************
 * 
 * The local user's cursor has moved.  Broadcast and update.
 *
 **************************************************************************/

void CursorOverlay::move(const Event &e) {
  char s[80];
  moveCursor( conference()->localID(), (int)e.pointer_x(), (int)e.pointer_y());
  sprintf(s, "%d:%d:%d", conference()->localID(), (int)e.pointer_x(), 
	  (int)e.pointer_y());
  conference()->connections()->toAll( new StrMsgSender(CURSORMOVE, s) );
}

void CursorOverlay::drag(const Event &e) { move(e); }


/**************************************************************************
 * 
 * The local cursor has entered the glyph.  Turn the real X cursor off
 * if "localRealCursor" is false (actually, turn it into a space character).
 *
 **************************************************************************/

void CursorOverlay::enter() {
  if (!localRealCursor) {
    window_cursor_ = patch_->canvas()->window()->cursor();
    patch_->canvas()->window()->cursor( 
	     new Cursor( WidgetKit::instance()->font(), ' ', ' '));
  }
}


/**************************************************************************
 * 
 * The local cursor has left the glyph.  Tell everyone else to turn off
 * the cursor (-1,-1).  Turn back on the real X cursor if we turned it
 * off when we entered.
 *
 **************************************************************************/

void CursorOverlay::leave() {
  char s[80];
  moveCursor( conference()->localID(), -1, -1);
  sprintf(s, "%d:%d:%d", conference()->localID(), -1, -1);
  conference()->connections()->toAll ( new StrMsgSender(CURSORMOVE, s) );
  if (!localRealCursor) 
    patch_->canvas()->window()->cursor( window_cursor_ );
}


